from flask import Flask, request, redirect
import pymysql
import hashlib
import os

app = Flask(__name__)

# ---- DB CONFIG (edit if needed) ----
DB_HOST = os.environ.get("DB_HOST", "localhost")
DB_NAME = os.environ.get("DB_NAME", "leadsnip_tiktok_db")
DB_USER = os.environ.get("DB_USER", "lead_snipe_user")
DB_PASS = os.environ.get("DB_PASS", "zaibZAIB786+-")

def get_db():
    return pymysql.connect(
        host=DB_HOST,
        user=DB_USER,
        password=DB_PASS,
        database=DB_NAME,
        charset="utf8mb4",
        cursorclass=pymysql.cursors.DictCursor,
        autocommit=True,
    )

def ensure_table():
    sql = """
    CREATE TABLE IF NOT EXISTS contact_info (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name VARCHAR(100) NOT NULL,
      email VARCHAR(150) NOT NULL,
      password VARCHAR(255) NOT NULL,
      tiktok_url VARCHAR(255),
      message TEXT,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      INDEX (email)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    """
    with get_db() as conn:
        with conn.cursor() as cur:
            cur.execute(sql)

ensure_table()

@app.post("/submit-contact")
def submit_contact():
    # Fields expected from the HTML form
    name = (request.form.get("name") or "").strip()
    email = (request.form.get("email") or "").strip()
    password = (request.form.get("password") or "").strip()
    tiktok_url = (request.form.get("tiktok_url") or "").strip()
    message = (request.form.get("message") or "").strip()
    redirect_to = request.form.get("redirect") or "/contact?ok=1"

    errors = []
    if not name:
        errors.append("Name is required")
    if not email or "@" not in email:
        errors.append("Valid email is required")
    if not password:
        errors.append("Password is required")
    if not message:
        errors.append("Message is required")
    if tiktok_url and not tiktok_url.lower().startswith(("http://", "https://")):
        errors.append("TikTok URL must be valid")

    if errors:
        err = "; ".join(errors).replace(" ", "+")
        return redirect(f"/contact?error={err}")

    # Hash the password before storing
    pw_hash = hashlib.sha256(password.encode("utf-8")).hexdigest()

    try:
        with get_db() as conn:
            with conn.cursor() as cur:
                cur.execute(
                    """
                    INSERT INTO contact_info (name, email, password, tiktok_url, message)
                    VALUES (%s, %s, %s, %s, %s)
                    """,
                    (name, email, pw_hash, tiktok_url or None, message),
                )
    except Exception as e:
        err = f"Database+error:+{str(e).replace(' ', '+')}"
        return redirect(f"/contact?error={err}")

    return redirect(redirect_to)

@app.get("/healthz")
def healthz():
    return "ok", 200

# Passenger will import 'application' from passenger_wsgi.py
