# /home/leadsnip/public_html/tiktok.leadsniper.online/app.py
from flask import (
    Flask, render_template, render_template_string, request,
    send_file, redirect, url_for, abort, Response, make_response
)
import os, sys, uuid, shutil, re, logging, glob, subprocess
from urllib.parse import urlparse
from typing import Optional, Dict, Any, Tuple, List

# Optional DB (PyMySQL). If unavailable, contact API will soft-disable.
try:
    import pymysql
    from pymysql.cursors import DictCursor
    HAVE_DB = True
except Exception:
    HAVE_DB = False

# Optional password hashing (used only if a password is posted AND table supports it)
try:
    from werkzeug.security import generate_password_hash
    HAVE_HASH = True
except Exception:
    HAVE_HASH = False

# ================== PATHS / APP / LOGGING ==================
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
TEMPLATES_DIR = os.path.join(BASE_DIR, "templates")
DOWNLOAD_DIR = os.path.join(BASE_DIR, "downloads")
LOG_DIR = os.path.join(BASE_DIR, "logs")
os.makedirs(DOWNLOAD_DIR, exist_ok=True)
os.makedirs(LOG_DIR, exist_ok=True)

LOG_FILE = os.path.join(LOG_DIR, "app.log")
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s %(levelname)s %(name)s: %(message)s",
    handlers=[logging.FileHandler(LOG_FILE), logging.StreamHandler(sys.stdout)],
    force=True,
)
log = logging.getLogger("tiktok")

# Explicit template folder (robust on shared hosts)
app = Flask(__name__, template_folder=TEMPLATES_DIR)
# Tolerate missing/extra trailing slashes, so /fr and /fr/ both work
app.url_map.strict_slashes = False

# ================== DB CONFIG (env-driven) ==================
DB_HOST = os.environ.get("DB_HOST", "localhost")
DB_USER = os.environ.get("DB_USER", "YOUR_DB_USERNAME")
DB_PASS = os.environ.get("DB_PASS", "YOUR_DB_PASSWORD")
DB_NAME = os.environ.get("DB_NAME", "leadsnip_tiktok_db")
CONTACT_TABLE = os.environ.get("CONTACT_TABLE", "contact_info")

def get_db():
    if not HAVE_DB:
        raise RuntimeError("PyMySQL not installed, contact API disabled.")
    return pymysql.connect(
        host=DB_HOST, user=DB_USER, password=DB_PASS, database=DB_NAME,
        cursorclass=DictCursor, autocommit=True, charset="utf8mb4"
    )

def table_has_column(column: str) -> bool:
    if not HAVE_DB:
        return False
    try:
        conn = get_db()
        with conn.cursor() as cur:
            cur.execute(f"SHOW COLUMNS FROM `{CONTACT_TABLE}` LIKE %s", (column,))
            row = cur.fetchone()
        conn.close()
        return bool(row)
    except Exception as e:
        log.warning("Unable to inspect table columns: %s", e)
        return False

def ensure_table():
    if not HAVE_DB:
        log.info("DB not available; skipping ensure_table.")
        return
    ddl = f"""
    CREATE TABLE IF NOT EXISTS `{CONTACT_TABLE}` (
        `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        `name` VARCHAR(191) NOT NULL,
        `email` VARCHAR(191) NOT NULL,
        `password` VARCHAR(255) NULL,
        `tiktok_url` VARCHAR(512) NULL,
        `message` TEXT NOT NULL,
        `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        INDEX (`email`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    """
    try:
        conn = get_db()
        with conn.cursor() as cur:
            cur.execute(ddl)
        conn.close()
        log.info("Ensured table `%s` exists.", CONTACT_TABLE)
    except Exception as e:
        log.exception("Failed creating/ensuring table: %s", e)

def drop_unique_email_if_exists():
    """MySQL: drop any UNIQUE index that targets the `email` column on CONTACT_TABLE."""
    if not HAVE_DB:
        return
    try:
        conn = get_db()
        with conn.cursor() as cur:
            cur.execute(f"SHOW INDEX FROM `{CONTACT_TABLE}`")
            rows = cur.fetchall() or []
            unique_by_key = {}
            for row in rows:
                non_unique = row.get("Non_unique", 1)
                key_name = row.get("Key_name")
                col_name = row.get("Column_name", "").lower()
                if int(non_unique) == 0 and key_name:
                    unique_by_key.setdefault(key_name, set()).add(col_name)

            for key_name, cols in unique_by_key.items():
                if "email" in cols:
                    try:
                        cur.execute(f"ALTER TABLE `{CONTACT_TABLE}` DROP INDEX `{key_name}`")
                        log.info("Dropped UNIQUE index `%s` on %s(email)", key_name, CONTACT_TABLE)
                    except Exception as e:
                        log.warning("Could not drop UNIQUE index %s: %s", key_name, e)
        conn.close()
    except Exception as e:
        log.warning("drop_unique_email_if_exists failed: %s", e)

if HAVE_DB:
    ensure_table()
    drop_unique_email_if_exists()

# ================== FFMPEG DETECTION ==================
def _detect_ffmpeg_dir() -> Optional[str]:
    """
    Return dir that contains ffmpeg or None to use system PATH.
    You can set export FFMPEG_DIR=/absolute/path/to/bin (dir that contains `ffmpeg`)
    """
    env_dir = os.environ.get("FFMPEG_DIR")
    if env_dir:
        ff = os.path.join(env_dir, "ffmpeg")
        if os.path.isfile(ff) and os.access(ff, os.X_OK):
            return env_dir
    local_ffmpeg = os.path.join(BASE_DIR, "bin", "ffmpeg")
    if os.path.isfile(local_ffmpeg) and os.access(local_ffmpeg, os.X_OK):
        return os.path.dirname(local_ffmpeg)
    if shutil.which("ffmpeg"):
        return None  # use PATH
    return None

FFMPEG_DIR = _detect_ffmpeg_dir()
log.info("FFMPEG_DIR resolved to: %s", FFMPEG_DIR or "system/None")

# ==================  ==================
try:
    import yt_dlp
    HAVE_YTDLP = True
except Exception as e:
    HAVE_YTDLP = False
    log.error("yt_dlp import failed: %s", e)

# ================== FALLBACK TEMPLATES ==================
INDEX_FALLBACK = """<!doctype html><meta charset="utf-8">
<title>TikTok Downloader</title>
<div style="max-width:720px;margin:48px auto;font-family:system-ui">
  <h1>TikTok Downloader</h1>
  <p>Paste a TikTok link and press Download.</p>
  <form method="post">
    <input type="url" name="url" placeholder="https://www.tiktok.com/@user/video/..." required
           style="width:100%;padding:12px;border:1px solid #ccc;border-radius:8px">
    <div style="margin-top:10px"><button style="padding:10px 16px;border:0;background:#3b82f6;color:#fff;border-radius:8px">Download</button></div>
  </form>
  {% if error %}<p style="color:#c00;font-weight:600">{{ error }}</p>{% endif %}
</div>"""

PREVIEW_FALLBACK = """<!doctype html><meta charset="utf-8">
<title>{{ title }} — Preview</title>
<div style="max-width:900px;margin:32px auto;font-family:system-ui">
  <h1>{{ title }}</h1>
  {% if file_size_mb %}<div>Size: {{ file_size_mb }} MB</div>{% endif %}
  <video controls style="width:100%;max-height:70vh;background:#000;margin-top:12px"
         src="{{ url_for('serve_video', file_id=file_id) }}"></video>
  <div style="margin-top:12px">
    <a href="{{ url_for('download_video', file_id=file_id) }}"
       style="padding:10px 16px;background:#3b82f6;color:#fff;border-radius:8px;text-decoration:none">Download MP4</a>
    <a href="{{ url_for('download_mp3', file_id=file_id) }}"
       style="padding:10px 16px;background:#10b981;color:#fff;border-radius:8px;text-decoration:none;margin-left:8px">Download MP3</a>
    <a href="{{ url_for('index') }}" style="margin-left:8px">Download another</a>
  </div>
  <audio controls style="width:100%;margin-top:12px">
    <source src="{{ url_for('serve_audio', file_id=file_id) }}" type="audio/mpeg">
  </audio>
</div>"""

def _render(name: str, **ctx):
    path = os.path.join(TEMPLATES_DIR, name)
    if os.path.isfile(path):
        return render_template(name, **ctx)
    if name == "index.html":
        return render_template_string(INDEX_FALLBACK, **ctx)
    if name == "preview.html":
        return render_template_string(PREVIEW_FALLBACK, **ctx)
    return render_template_string("<pre>{{ ctx|tojson }}</pre>", ctx=ctx)

# ================== HELPERS ==================
BAD_SUFFIXES = {".part", ".ytdl", ".temp", ".tmp", ".aria2", ".m3u8", ".json"}
def _file_glob(file_id: str) -> List[str]:
    return glob.glob(os.path.join(DOWNLOAD_DIR, f"{file_id}.*"))

def _is_final_candidate(path: str) -> bool:
    base, ext = os.path.splitext(path)
    return ext.lower() not in BAD_SUFFIXES and os.path.isfile(path) and os.path.getsize(path) > 0

def _find_final_path(file_id: str) -> Tuple[str, str]:
    """
    Return (absolute_path, ext) for the saved download, preferring .mp4 when available,
    and ignoring partial/temp files.
    """
    matches = [p for p in _file_glob(file_id) if _is_final_candidate(p)]
    if not matches:
        raise FileNotFoundError("Output file not finalized yet (no playable file).")
    mp4s = [m for m in matches if m.lower().endswith(".mp4")]
    path = mp4s[0] if mp4s else matches[0]
    ext = os.path.splitext(path)[1].lstrip(".").lower()
    return path, ext

def _is_tiktok_url(url: str) -> bool:
    try:
        host = urlparse(url).netloc.lower()
        return any(h in host for h in ("tiktok.com", "tt.tiktok.com", "vt.tiktok.com", "vm.tiktok.com", "akamaized.net", "byteoversea", "tiktokcdn.com"))
    except Exception:
        return False

# Byte range support for <video>/<audio> seek
_RANGE_RE = re.compile(r"bytes=(\d*)-(\d*)")

def _serve_with_range(path: str, mime: str = "video/mp4"):
    if not os.path.isfile(path):
        abort(404)
    size = os.path.getsize(path)
    rng = request.headers.get("Range", "")
    m = _RANGE_RE.match(rng) if rng else None

    if not m:
        with open(path, "rb") as f:
            data = f.read()
        resp = make_response(data)
        resp.headers["Content-Type"] = mime
        resp.headers["Content-Length"] = str(size)
        resp.headers["Accept-Ranges"] = "bytes"
        resp.headers["Cache-Control"] = "no-store, max-age=0"
        return resp

    start_s, end_s = m.groups()
    if start_s == "" and end_s == "":
        start, end = 0, size - 1
    elif start_s == "":
        length = max(int(end_s or 0), 0)
        start = max(size - length, 0)
        end = size - 1
    else:
        start = int(start_s)
        end = int(end_s) if end_s else size - 1
        if start >= size:
            resp = Response(status=416)
            resp.headers["Content-Range"] = f"bytes */{size}"
            return resp
    start = max(start, 0)
    end = min(end, size - 1)
    length = end - start + 1

    with open(path, "rb") as f:
        f.seek(start)
        data = f.read(length)

    resp = make_response(data, 206)
    resp.headers["Content-Type"] = mime
    resp.headers["Content-Range"] = f"bytes {start}-{end}/{size}"
    resp.headers["Content-Length"] = str(length)
    resp.headers["Accept-Ranges"] = "bytes"
    resp.headers["Cache-Control"] = "no-store, max-age=0"
    return resp

def _send_file_safe(path: str, as_attachment=False, download_name=None, mimetype=None, conditional=False):
    try:
        return send_file(path, as_attachment=as_attachment, download_name=download_name,
                         mimetype=mimetype, conditional=conditional)
    except TypeError:
        return send_file(path, as_attachment=as_attachment, attachment_filename=download_name,
                         mimetype=mimetype)

# ==================  OPTIONS ==================
def _ydl_opts(outtmpl: str) -> Dict[str, Any]:
    """
    Prefer final MP4, but gracefully avoid FFmpeg when it's missing.
    Adds TikTok cookie support and stricter headers.
    """
    cookies_path = os.path.join(BASE_DIR, "cookies.txt")
    have_cookies = os.path.isfile(cookies_path)

    headers = {
        "User-Agent": (
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) "
            "AppleWebKit/537.36 (KHTML, like Gecko) "
            "Chrome/120.0.0.0 Safari/537.36"
        ),
        "Accept-Language": "en-US,en;q=0.9",
        "Referer": "https://www.tiktok.com/",
    }

    ffmpeg_present = (FFMPEG_DIR is not None) or bool(shutil.which("ffmpeg"))

    # If FFmpeg is present, we can remux/faststart. If not, prefer direct MP4 to avoid remux.
    if ffmpeg_present:
        fmt = "best[ext=mp4][acodec!=none]/bestvideo[ext=mp4]+bestaudio/best"
        postprocessors = [{"key": "FFmpegVideoConvertor", "preferedformat": "mp4"}]
        post_args = ["-movflags", "faststart"]
        merge_fmt = "mp4"
        final_ext = "mp4"
    else:
        # Avoid remux/postprocess; ask for a ready MP4 when possible
        fmt = "mp4[acodec!=none]/best[ext=mp4]/best"
        postprocessors = []
        post_args = []
        merge_fmt = None
        final_ext = None

    opts: Dict[str, Any] = {
        "format": fmt,
        "outtmpl": outtmpl,
        "prefer_ffmpeg": True,
        "ffmpeg_location": FFMPEG_DIR,   # None => PATH
        "postprocessors": postprocessors,
        "postprocessor_args": post_args,
        "quiet": True,
        "noplaylist": True,
        "retries": 3,
        "fragment_retries": 3,
        "concurrent_fragment_downloads": 1,
        "http_headers": headers,
        "socket_timeout": 30,
        "geo_bypass_country": "US",
        "nocheckcertificate": os.environ.get("YTDLP_INSECURE") == "1",
    }
    if merge_fmt:
        opts["merge_output_format"] = merge_fmt
    if final_ext:
        opts["final_ext"] = final_ext
    if have_cookies:
        opts["cookiefile"] = cookies_path
    return opts

# ================== MP3 CONVERSION HELPERS ==================
def _ensure_mp3(file_id: str) -> str:
    """
    Ensure an MP3 exists for the given file_id.
    Converts the already-downloaded video to MP3 using FFmpeg.
    Returns absolute path to the MP3.
    """
    src_path, _ext = _find_final_path(file_id)  # raises if missing
    mp3_path = os.path.join(DOWNLOAD_DIR, f"{file_id}.mp3")

    # If already converted and non-empty, reuse it
    if os.path.isfile(mp3_path) and os.path.getsize(mp3_path) > 0:
        return mp3_path

    # Require ffmpeg
    ffmpeg_bin = os.path.join(FFMPEG_DIR, "ffmpeg") if FFMPEG_DIR else shutil.which("ffmpeg")
    if not ffmpeg_bin or (ffmpeg_bin != "ffmpeg" and not os.path.isfile(ffmpeg_bin)):
        raise RuntimeError("FFmpeg not found. Install FFmpeg or set $FFMPEG_DIR to its bin directory.")

    # Clean convert (overwrite any broken file)
    try:
        cmd = [
            ffmpeg_bin, "-hide_banner", "-nostdin", "-y",
            "-i", src_path,
            "-vn", "-acodec", "libmp3lame", "-q:a", "2",
            mp3_path
        ]
        log.info("Converting to MP3: %s", " ".join(cmd))
        res = subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True)
        if not os.path.isfile(mp3_path) or os.path.getsize(mp3_path) == 0:
            raise RuntimeError("FFmpeg did not produce a valid MP3 file.")
        return mp3_path
    except subprocess.CalledProcessError as e:
        err = (e.stderr or b"").decode("utf-8", errors="ignore")[:1000]
        raise RuntimeError(f"FFmpeg conversion failed: {err}")

# ================== SHARED DOWNLOAD HANDLER ==================
def _handle_download_post(redirect_preview_endpoint: str, template_name_for_error: str):
    """
    Run the same download logic, but redirect to the given preview endpoint
    (e.g., 'preview' for EN or 'fr_preview' for FR). If there's an error,
    re-render the given template with an error message.
    """
    url = (request.form.get("url") or "").strip()
    if not url or not _is_tiktok_url(url):
        return _render(template_name_for_error, error="Please paste a valid TikTok link (e.g. https://www.tiktok.com/@user/video/123...)."), 400
    if not HAVE_YTDLP:
        return _render(template_name_for_error, error="Server is missing . Please install it."), 500

    file_id = uuid.uuid4().hex
    out_pattern = os.path.join(DOWNLOAD_DIR, f"{file_id}.%(ext)s")
    ydl_opts = _ydl_opts(out_pattern)

    try:
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            info = ydl.extract_info(url, download=True)

        path, ext = _find_final_path(file_id)
        if ext != "mp4":
            log.warning("Final ext is %s; attempting to continue anyway.", ext)
        if not os.path.isfile(path) or os.path.getsize(path) == 0:
            raise FileNotFoundError("Finalized MP4 not found or empty. Check FFmpeg installation.")

        title = (info or {}).get("title", "TikTok Video")
        log.info("Saved video OK: %s (%s, %.2f MB)", path, ext, os.path.getsize(path)/(1024*1024))
        return redirect(url_for(redirect_preview_endpoint, file_id=file_id, title=title))

    except Exception as err:
        log.exception("Download failed")
        hint = ""
        if FFMPEG_DIR is None and not shutil.which("ffmpeg"):
            hint = " (FFmpeg not found — install FFmpeg or set $FFMPEG_DIR to its bin dir)"
        return _render(template_name_for_error, error=f"Download error: {err}{hint}"), 500

# ================== ROUTES (EN default) ==================
@app.route("/healthz")
def healthz():
    return "ok", 200

@app.route("/", methods=["GET", "POST"])
def index():
    if request.method == "POST":
        return _handle_download_post(redirect_preview_endpoint="preview",
                                     template_name_for_error="index.html")
    return _render("index.html")

@app.route("/preview/<file_id>")
def preview(file_id):
    title = request.args.get("title", "TikTok Video")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("preview.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


@app.route("/preview", methods=["GET"])
def preview_from_query():
    """Accepts ?url=..., runs the download, then redirects to /preview/<file_id>."""
    url = (request.args.get("url") or "").strip()
    if not url or not _is_tiktok_url(url):
        # show the same homepage with an error
        return _render("index.html", error="Please paste a valid TikTok link."), 400
    if not HAVE_YTDLP:
        return _render("index.html", error="Server is missing . Please install it."), 500

    file_id = uuid.uuid4().hex
    out_pattern = os.path.join(DOWNLOAD_DIR, f"{file_id}.%(ext)s")
    ydl_opts = _ydl_opts(out_pattern)

    try:
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            info = ydl.extract_info(url, download=True)

        # ensure the file exists
        _path, _ext = _find_final_path(file_id)

        title = (info or {}).get("title", "TikTok Video")
        return redirect(url_for("preview", file_id=file_id, title=title))

    except Exception as err:
        log.exception("Download via /preview?url failed")
        hint = ""
        if FFMPEG_DIR is None and not shutil.which("ffmpeg"):
            hint = " (FFmpeg not found — install FFmpeg or set $FFMPEG_DIR)"
        return _render("index.html", error=f"Download error: {err}{hint}"), 500


@app.route("/video/<file_id>", methods=["GET", "HEAD"])
def serve_video(file_id):
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    mime = "video/mp4"
    if request.method == "HEAD":
        size = os.path.getsize(path)
        resp = Response(status=200)
        resp.headers["Content-Type"] = mime
        resp.headers["Content-Length"] = str(size)
        resp.headers["Accept-Ranges"] = "bytes"
        resp.headers["Cache-Control"] = "no-store, max-age=0"
        return resp
    return _serve_with_range(path, mime=mime)

@app.route("/download/<file_id>")
def download_video(file_id):
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    return _send_file_safe(path, as_attachment=True, download_name=f"{file_id}.mp4", mimetype="video/mp4")

# ======== NEW AUDIO ROUTES ========
@app.route("/audio/<file_id>", methods=["GET", "HEAD"])
def serve_audio(file_id):
    """
    Stream the MP3 with byte-range support (useful for <audio> players).
    """
    try:
        mp3_path = _ensure_mp3(file_id)
    except Exception:
        log.exception("MP3 stream failed")
        abort(404)
    mime = "audio/mpeg"
    if request.method == "HEAD":
        size = os.path.getsize(mp3_path)
        resp = Response(status=200)
        resp.headers["Content-Type"] = mime
        resp.headers["Content-Length"] = str(size)
        resp.headers["Accept-Ranges"] = "bytes"
        resp.headers["Cache-Control"] = "no-store, max-age=0"
        return resp
    return _serve_with_range(mp3_path, mime=mime)

@app.route("/download-mp3/<file_id>")
def download_mp3(file_id):
    """
    Force-download the MP3 as an attachment.
    """
    try:
        mp3_path = _ensure_mp3(file_id)
    except Exception:
        log.exception("MP3 download failed")
        abort(404)
    return _send_file_safe(mp3_path, as_attachment=True,
                           download_name=f"{file_id}.mp3",
                           mimetype="audio/mpeg")

# ---- Extra content pages your navs link to (EN) ----
@app.route("/tiktok-downloader", methods=["GET", "POST"])
def tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(redirect_preview_endpoint="preview",
                                     template_name_for_error="TiktokVideoDownloader.html")
    return _render("TiktokVideoDownloader.html")

@app.route("/download-tiktok-mp3", methods=["GET", "POST"])
def download_tiktok_mp3():
    # (Optional) if you add MP3 extraction page/UI, render template here.
    # Current flow: MP3 generated on-demand from /download-mp3/<file_id> or /audio/<file_id>
    return _render("DownloadTikTokMP3.html")

@app.route("/about")
def about():
    return _render("about.html")

@app.route("/contact")
def contact():
    return _render("contact.html")

@app.route("/privacy")
def privacy():
    return _render("privacy.html")

@app.route("/terms")
def terms():
    return _render("terms.html")

# ================== CONTACT API (shared) ==================
@app.route("/contact_api/submit-contact", methods=["POST"])
def submit_contact():
    # Prefer explicit redirect field; else decide based on where the POST came from (EN/FR)
    redirect_to = request.form.get("redirect")
    if not redirect_to:
        # If path starts with /fr, default to FR contact page; else EN.
        if request.path.startswith("/fr") or (request.referrer and "/fr" in request.referrer):
            redirect_to = "/fr/contact?ok=1"
        else:
            redirect_to = url_for("contact") + "?ok=1"

    name = (request.form.get("name") or "").strip()
    email = (request.form.get("email") or "").strip()
    password = (request.form.get("password") or "").strip()
    tiktok_url = (request.form.get("tiktok_url") or "").strip()
    topic = (request.form.get("topic") or "").strip()
    message = (request.form.get("message") or "").strip()

    if not name or not email or not message:
        # Keep language based on referrer if we can
        if request.referrer and "/fr" in request.referrer:
            return redirect("/fr/contact?error=" + "Champs+obligatoires+manquants+(nom,+e-mail,+message).")
        return redirect(url_for("contact") + "?error=" + "Missing+required+fields+(name,+email,+message).")

    if topic:
        message_to_store = f"[Topic: {topic}]\n{message}"
    else:
        message_to_store = message

    if not HAVE_DB:
        return redirect(redirect_to)

    try:
        has_pass_col = table_has_column("password")
        if has_pass_col and password:
            hashed = generate_password_hash(password) if HAVE_HASH else password
            sql = f"""
                INSERT INTO `{CONTACT_TABLE}` (`name`, `email`, `password`, `tiktok_url`, `message`)
                VALUES (%s, %s, %s, %s, %s)
            """
            params = (name, email, hashed, tiktok_url or None, message_to_store)
        else:
            sql = f"""
                INSERT INTO `{CONTACT_TABLE}` (`name`, `email`, `tiktok_url`, `message`)
                VALUES (%s, %s, %s, %s)
            """
            params = (name, email, tiktok_url or None, message_to_store)

        conn = get_db()
        with conn.cursor() as cur:
            cur.execute(sql, params)
        conn.close()
        return redirect(redirect_to)

    except Exception as e:
        log.exception("Contact submit failed")
        # Keep language based on referrer if possible
        msg = str(e).replace(" ", "+")
        if request.referrer and "/fr" in request.referrer:
            return redirect("/fr/contact?error=" + msg)
        return redirect(url_for("contact") + "?error=" + msg)

# ================== FR ROUTES (prefix under /fr/*) ==================
# Home (GET + POST so FR stays on FR preview)
@app.route("/fr/", methods=["GET", "POST"])
def fr_home():
    if request.method == "POST":
        return _handle_download_post(redirect_preview_endpoint="fr_preview",
                                     template_name_for_error="fr/index_fr.html")
    return _render("fr/index_fr.html")

# (Optional) /fr -> /fr/ convenience
@app.route("/fr")
def fr_home_redirect():
    return redirect("/fr/")

# Content pages
@app.route("/fr/tiktok-downloader", methods=["GET", "POST"])
def fr_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(redirect_preview_endpoint="fr_preview",
                                     template_name_for_error="fr/TiktokVideoDownloader_fr.html")
    return _render("fr/TiktokVideoDownloader_fr.html")

@app.route("/fr/download-tiktok-mp3", methods=["GET", "POST"])
def fr_download_tiktok_mp3():
    return _render("fr/DownloadTikTokMP3_fr.html")

@app.route("/fr/about")
def fr_about():
    return _render("fr/about_fr.html")

@app.route("/fr/contact", methods=["GET"])
def fr_contact():
    return _render("fr/contact_fr.html")

@app.route("/fr/privacy")
def fr_privacy():
    return _render("fr/privacy_fr.html")

@app.route("/fr/terms")
def fr_terms():
    return _render("fr/terms_fr.html")

# Preview (FR view of the same downloaded file)
@app.route("/fr/preview/<file_id>")
def fr_preview(file_id):
    # Accept title from query if provided (e.g., when redirected after download)
    title = request.args.get("title", "Titre de la vidéo")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("fr/preview_fr.html", file_id=file_id, title=title, file_size_mb=file_size_mb)

# Allow FR contact form to reuse the same backend handler:
# Your FR template can post to /fr/contact/submit OR /contact_api/submit-contact
app.add_url_rule("/fr/contact/submit", view_func=submit_contact, methods=["POST"])








# ----------------------
# Arabic (ar) routes
# ----------------------

@app.route("/ar/", methods=["GET", "POST"])
def ar_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="ar_preview",
            template_name_for_error="ar/index_ar.html"
        )
    return _render("ar/index_ar.html")

# Convenience: /ar -> /ar/
@app.route("/ar")
def ar_home_redirect():
    return redirect("/ar/")

# Content pages
@app.route("/ar/tiktok-downloader", methods=["GET", "POST"])
def ar_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="ar_preview",
            template_name_for_error="ar/TiktokVideoDownloader_ar.html"
        )
    return _render("ar/TiktokVideoDownloader_ar.html")

@app.route("/ar/download-tiktok-mp3", methods=["GET", "POST"])
def ar_download_tiktok_mp3():
    # (If you ever need POST handling here, mirror ar_tiktok_downloader)
    return _render("ar/DownloadTikTokMP3_ar.html")

# Preview (AR view of the same downloaded file)
@app.route("/ar/preview/<file_id>")
def ar_preview(file_id):
    # Accept title from query if provided
    title = request.args.get("title", "عنوان الفيديو")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("ar/preview_ar.html", file_id=file_id, title=title, file_size_mb=file_size_mb)
    
    
    
    
    
    
    
    
    
    
    
    
    # =========================
# Japanese (ja)
# =========================

@app.route("/ja/", methods=["GET", "POST"])
def ja_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="ja_preview",
            template_name_for_error="ja/index_ja.html"
        )
    return _render("ja/index_ja.html")

@app.route("/ja")
def ja_home_redirect():
    return redirect("/ja/")

@app.route("/ja/tiktok-downloader", methods=["GET", "POST"])
def ja_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="ja_preview",
            template_name_for_error="ja/TiktokVideoDownloader_ja.html"
        )
    return _render("ja/TiktokVideoDownloader_ja.html")

@app.route("/ja/download-tiktok-mp3", methods=["GET", "POST"])
def ja_download_tiktok_mp3():
    return _render("ja/DownloadTikTokMP3_ja.html")

@app.route("/ja/preview/<file_id>")
def ja_preview(file_id):
    title = request.args.get("title", "動画のタイトル")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("ja/preview_ja.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


# =========================
# Thai (th)
# =========================

@app.route("/th/", methods=["GET", "POST"])
def th_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="th_preview",
            template_name_for_error="th/index_th.html"
        )
    return _render("th/index_th.html")

@app.route("/th")
def th_home_redirect():
    return redirect("/th/")

@app.route("/th/tiktok-downloader", methods=["GET", "POST"])
def th_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="th_preview",
            template_name_for_error="th/TiktokVideoDownloader_th.html"
        )
    return _render("th/TiktokVideoDownloader_th.html")

@app.route("/th/download-tiktok-mp3", methods=["GET", "POST"])
def th_download_tiktok_mp3():
    return _render("th/DownloadTikTokMP3_th.html")

@app.route("/th/preview/<file_id>")
def th_preview(file_id):
    title = request.args.get("title", "ชื่อวิดีโอ")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("th/preview_th.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


# =========================
# Indonesian (id)
# =========================

@app.route("/id/", methods=["GET", "POST"])
def id_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="id_preview",
            template_name_for_error="id/index_id.html"
        )
    return _render("id/index_id.html")

@app.route("/id")
def id_home_redirect():
    return redirect("/id/")

@app.route("/id/tiktok-downloader", methods=["GET", "POST"])
def id_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="id_preview",
            template_name_for_error="id/TiktokVideoDownloader_id.html"
        )
    return _render("id/TiktokVideoDownloader_id.html")

@app.route("/id/download-tiktok-mp3", methods=["GET", "POST"])
def id_download_tiktok_mp3():
    return _render("id/DownloadTikTokMP3_id.html")

@app.route("/id/preview/<file_id>")
def id_preview(file_id):
    title = request.args.get("title", "Judul video")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("id/preview_id.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


# =========================
# German (du + de aliases)
# =========================
# Uses templates/du/... to match your chosen folder name.

@app.route("/du/", methods=["GET", "POST"])
@app.route("/de/", methods=["GET", "POST"])
def du_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="du_preview",
            template_name_for_error="du/index_du.html"
        )
    return _render("du/index_du.html")

@app.route("/du")
@app.route("/de")
def du_home_redirect():
    return redirect("/du/")  # canonicalize to /du/

@app.route("/du/tiktok-downloader", methods=["GET", "POST"])
@app.route("/de/tiktok-downloader", methods=["GET", "POST"])
def du_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="du_preview",
            template_name_for_error="du/TiktokVideoDownloader_du.html"
        )
    return _render("du/TiktokVideoDownloader_du.html")

@app.route("/du/download-tiktok-mp3", methods=["GET", "POST"])
@app.route("/de/download-tiktok-mp3", methods=["GET", "POST"])
def du_download_tiktok_mp3():
    return _render("du/DownloadTikTokMP3_du.html")

@app.route("/du/preview/<file_id>")
@app.route("/de/preview/<file_id>")
def du_preview(file_id):
    title = request.args.get("title", "Videotitel")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("du/preview_du.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


# =========================
# Spanish (es)
# =========================

@app.route("/es/", methods=["GET", "POST"])
def es_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="es_preview",
            template_name_for_error="es/index_es.html"
        )
    return _render("es/index_es.html")

@app.route("/es")
def es_home_redirect():
    return redirect("/es/")

@app.route("/es/tiktok-downloader", methods=["GET", "POST"])
def es_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="es_preview",
            template_name_for_error="es/TiktokVideoDownloader_es.html"
        )
    return _render("es/TiktokVideoDownloader_es.html")

@app.route("/es/download-tiktok-mp3", methods=["GET", "POST"])
def es_download_tiktok_mp3():
    return _render("es/DownloadTikTokMP3_es.html")

@app.route("/es/preview/<file_id>")
def es_preview(file_id):
    title = request.args.get("title", "Título del video")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("es/preview_es.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


# =========================
# Turkish (tr)
# =========================

@app.route("/tr/", methods=["GET", "POST"])
def tr_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="tr_preview",
            template_name_for_error="tr/index_tr.html"
        )
    return _render("tr/index_tr.html")

@app.route("/tr")
def tr_home_redirect():
    return redirect("/tr/")

@app.route("/tr/tiktok-downloader", methods=["GET", "POST"])
def tr_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="tr_preview",
            template_name_for_error="tr/TiktokVideoDownloader_tr.html"
        )
    return _render("tr/TiktokVideoDownloader_tr.html")

@app.route("/tr/download-tiktok-mp3", methods=["GET", "POST"])
def tr_download_tiktok_mp3():
    return _render("tr/DownloadTikTokMP3_tr.html")

@app.route("/tr/preview/<file_id>")
def tr_preview(file_id):
    title = request.args.get("title", "Video başlığı")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("tr/preview_tr.html", file_id=file_id, title=title, file_size_mb=file_size_mb)


# =========================
# Italian (it)
# =========================

@app.route("/it/", methods=["GET", "POST"])
def it_home():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="it_preview",
            template_name_for_error="it/index_it.html"
        )
    return _render("it/index_it.html")

@app.route("/it")
def it_home_redirect():
    return redirect("/it/")

@app.route("/it/tiktok-downloader", methods=["GET", "POST"])
def it_tiktok_downloader():
    if request.method == "POST":
        return _handle_download_post(
            redirect_preview_endpoint="it_preview",
            template_name_for_error="it/TiktokVideoDownloader_it.html"
        )
    return _render("it/TiktokVideoDownloader_it.html")

@app.route("/it/download-tiktok-mp3", methods=["GET", "POST"])
def it_download_tiktok_mp3():
    return _render("it/DownloadTikTokMP3_it.html")

@app.route("/it/preview/<file_id>")
def it_preview(file_id):
    title = request.args.get("title", "Titolo del video")
    try:
        path, ext = _find_final_path(file_id)
    except Exception:
        abort(404)
    file_size_mb = round(os.path.getsize(path) / (1024 * 1024), 2)
    return _render("it/preview_it.html", file_id=file_id, title=title, file_size_mb=file_size_mb)

    
    
    
    
    
    
    
    
    
    
    


# ================== DEBUG ==================
@app.route("/debug")
def debug():
    """Lightweight diagnostics page."""
    info = {}
    try:
        import yt_dlp
        info["yt_dlp_version"] = getattr(yt_dlp, "__version__", "unknown")
    except Exception as e:
        info["yt_dlp_import_error"] = str(e)

    info["FFMPEG_DIR"] = FFMPEG_DIR or ""
    info["ffmpeg_in_path"] = bool(shutil.which("ffmpeg"))
    info["cookies_txt_exists"] = os.path.isfile(os.path.join(BASE_DIR, "cookies.txt"))
    info["downloads_writable"] = os.access(DOWNLOAD_DIR, os.W_OK)
    info["cwd"] = os.getcwd()
    try:
        size = len(os.listdir(DOWNLOAD_DIR))
    except Exception:
        size = "ERR"
    info["downloads_count"] = size

    return make_response({"ok": True, "info": info}, 200)

if __name__ == "__main__":
    # For local testing; on cPanel/Passenger this is usually ignored
    app.run(host="0.0.0.0", port=int(os.environ.get("PORT", 5000)), debug=False)