<?php
require __DIR__.'/config.php';

$pdo = db();

// --- Input ---
$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) { http_response_code(404); echo "Not found."; exit; }

// --- Try phones first (original table) ---
$stmt = $pdo->prepare("
  SELECT
    p.id,
    p.source_url,
    p.remote_image_url,
    p.local_image_path,
    p.price_text,
    p.price_rs,
    p.intro,
    p.mapping_json,
    p.specs_json,
    p.specs_table_md,
    p.created_at,
    NULL AS title,
    NULL AS price_usd,
    NULL AS table_html,
    NULL AS description,
    'phones' AS _src
  FROM phones p
  WHERE p.id = :id
");
$stmt->execute([':id' => $id]);
$row = $stmt->fetch(PDO::FETCH_ASSOC);

// --- Fallback: whatmobile_products (new table) ---
if (!$row) {
  $stmt = $pdo->prepare("
    SELECT
      w.id,
      w.url            AS source_url,
      NULL             AS remote_image_url,
      w.image_path     AS local_image_path,
      NULL             AS price_text,
      -- price_rs comes as VARCHAR; coerce digits only
      CAST(NULLIF(REGEXP_REPLACE(w.price_rs, '[^0-9]', ''), '') AS UNSIGNED) AS price_rs,
      w.description    AS intro,
      NULL             AS mapping_json,
      NULL             AS specs_json,
      NULL             AS specs_table_md,
      NULL             AS created_at,
      w.title          AS title,
      w.price_usd      AS price_usd,
      w.table_html     AS table_html,
      w.description    AS description,
      'whatmobile_products' AS _src
    FROM whatmobile_products w
    WHERE w.id = :id
  ");
  $stmt->execute([':id' => $id]);
  $row = $stmt->fetch(PDO::FETCH_ASSOC);
}

if (!$row) { http_response_code(404); echo "Not found."; exit; }

// ---- Helper fallbacks (in case not globally defined) ----
if (!function_exists('derive_title')) {
  function derive_title(array $row): string {
    $t = trim((string)($row['title'] ?? ''));
    if ($t !== '') return $t;
    $intro = trim((string)($row['intro'] ?? ''));
    if ($intro !== '') {
      // if mb_strimwidth missing, use substr
      if (function_exists('mb_strimwidth')) return mb_strimwidth(strip_tags($intro), 0, 80, '…', 'UTF-8');
      $plain = strip_tags($intro);
      return strlen($plain) > 80 ? substr($plain, 0, 77).'…' : $plain;
    }
    $src = trim((string)($row['source_url'] ?? ''));
    return $src ? basename(parse_url($src, PHP_URL_PATH) ?? 'Phone') : 'Phone';
  }
}
if (!function_exists('product_image_url')) {
  function product_image_url(array $row): ?string {
    $local  = trim((string)($row['local_image_path'] ?? ''));
    $remote = trim((string)($row['remote_image_url'] ?? ''));
    return $local ?: ($remote ?: null);
  }
}
if (!function_exists('money')) {
  function money(array $row): string {
    $p = $row['price_rs'] ?? null;
    if ($p === null || $p === '') return '—';
    $p = (int)$p;
    return 'Rs '.number_format($p);
  }
}

// ---- Normalize for the template ----
$title  = derive_title($row);
$imgUrl = product_image_url($row);
$moneyStr  = money($row);

// mapping_json/specs_json may be missing in whatmobile_products
$kv = [];
if (!empty($row['mapping_json'])) {
  $kv = json_decode($row['mapping_json'], true);
  if (!is_array($kv)) $kv = [];
}

$records = [];
if (!empty($row['specs_json'])) {
  $records = json_decode($row['specs_json'], true);
  if (!is_array($records)) $records = [];
}

// Build dynamic columns for records table
$cols = [];
foreach ($records as $rec) {
  foreach ((array)$rec as $ck=>$cv) $cols[$ck] = true;
}
$cols = array_keys($cols);

function pretty_key(string $k): string {
  // common prettifications
  $map = [
    'ram'=>'RAM','storage'=>'Storage','battery'=>'Battery','display'=>'Display',
    'chipset'=>'Chipset','os'=>'OS','camera'=>'Camera','frontcamera'=>'Front Camera',
    'rearcamera'=>'Rear Camera','dimensions'=>'Dimensions','weight'=>'Weight','colors'=>'Colors'
  ];
  if (isset($map[$k])) return $map[$k];
  // insert spaces between alpha/num boundaries
  $k = preg_replace('~([a-z])([A-Z0-9])~', '$1 $2', $k);
  $k = preg_replace('~([0-9])([a-zA-Z])~', '$1 $2', $k);
  $k = ucwords(str_replace('_',' ', $k));
  return $k;
}

$addedAt = $row['created_at'] ?? null; // may be NULL for whatmobile_products
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title><?=h($title)?> — Preview</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    :root { --gap:16px; --radius:14px; --line:#e5e7eb; --muted:#6b7280; --shade:#f7f8fb; --text:#111827; }
    *{box-sizing:border-box}
    body{margin:0; font:15px/1.55 system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial; color:var(--text); background:#fff;}
    a{color:#111827; text-decoration:none}
    header{border-bottom:1px solid var(--line); padding:12px 16px;}
    .wrap{max-width:1100px; margin:0 auto; padding:0 12px;}
    .crumbs{font-size:13px; color:var(--muted);}
    .top{display:grid; grid-template-columns: 380px 1fr; gap:20px; padding:18px 0;}
    @media (max-width:900px){ .top{ grid-template-columns:1fr; } }
    .media{border:1px solid var(--line); border-radius: var(--radius); background:#fff; overflow:hidden;}
    .media .ph{aspect-ratio:1.2/1; background:#fff;}
    .media img{width:100%; height:100%; object-fit:contain; display:block;}
    h1{font-size:24px; margin:0 0 6px;}
    .price{font-size:20px; font-weight:700; margin:8px 0 12px;}
    .meta{color:var(--muted); font-size:13px; margin-bottom:12px;}
    .btns{display:flex; gap:10px; margin:10px 0 18px;}
    .btn{display:inline-block; padding:10px 12px; border-radius:10px; border:1px solid var(--line);}
    .btn.primary{background:#111827; color:#fff; border-color:#111827;}
    .panel{border:1px solid var(--line); border-radius: var(--radius); padding:14px; background:#fff; margin:16px 0;}
    .panel h2{font-size:18px; margin:0 0 10px;}
    table.kv{width:100%; border-collapse:collapse; font-size:14px;}
    table.kv th, table.kv td{border-bottom:1px solid var(--line); padding:8px; vertical-align:top;}
    table.kv th{width:210px; text-align:left; color:var(--muted); font-weight:600;}
    .scroll{overflow:auto; border:1px solid var(--line); border-radius:12px; }
    table.rec{border-collapse:collapse; width:100%; min-width:700px;}
    table.rec th, table.rec td{border-bottom:1px solid var(--line); padding:8px; text-align:left; vertical-align:top; font-size:13px;}
    pre.code{background:var(--shade); padding:12px; border-radius:12px; overflow:auto; border:1px solid var(--line);}
    footer{padding:20px 0 40px; color:var(--muted); font-size:13px;}
    .specs-html table { width: 100%; border-collapse: collapse; }
    .specs-html table th, .specs-html table td { border: 1px solid var(--line); padding: 6px; font-size: 14px; vertical-align: top; }
    .specs-html table th { background: #f9fafb; text-align: left; }
  </style>
</head>
<body>
<header>
  <div class="wrap">
    <div class="crumbs"><a href="index.php">← Back to list</a></div>
  </div>
</header>

<main class="wrap" itemscope itemtype="https://schema.org/Product">
  <section class="top">
    <div class="media">
      <div class="ph">
        <?php if ($imgUrl): ?>
          <img src="<?=h($imgUrl)?>" alt="<?=h($title)?>" itemprop="image">
        <?php endif; ?>
      </div>
    </div>
    <div>
      <h1 itemprop="name"><?=h($title)?></h1>
      <div class="price" itemprop="offers" itemscope itemtype="https://schema.org/Offer">
        <span itemprop="priceCurrency" content="PKR"></span>
        <span itemprop="price"><?=h($moneyStr)?></span>
      </div>
      <div class="meta">
        <?php if (!empty($addedAt)): ?>
          Added: <?=h($addedAt)?>
        <?php else: ?>
          Added: —
        <?php endif; ?>
        <?php if (!empty($row['_src'])): ?>
          • Source table: <strong><?=h($row['_src'])?></strong>
        <?php endif; ?>
      </div>

      <?php if (!empty($row['intro'])): ?>
        <div class="panel">
          <h2>Overview</h2>
          <p itemprop="description"><?=nl2br(h($row['intro']))?></p>
        </div>
      <?php endif; ?>

      <?php if ($row['_src'] === 'whatmobile_products' && !empty($row['table_html'])): ?>
        <div class="panel specs-html">
          <h2>Specifications</h2>
          <!-- table_html comes from your crawler; if any user input can reach here, sanitize further -->
          <?= $row['table_html'] ?>
        </div>
      <?php endif; ?>

      <div class="btns">
        <?php if (!empty($row['source_url'])): ?>
          <a class="btn" href="<?=h($row['source_url'])?>" target="_blank" rel="noopener">Open Original</a>
        <?php endif; ?>
        <a class="btn primary" href="index.php">Browse More</a>
      </div>
    </div>
  </section>

  <?php if ($kv): ?>
  <section class="panel">
    <h2>Key Specifications</h2>
    <table class="kv">
      <tbody>
        <?php foreach ($kv as $k=>$v): if ($v==='') continue; ?>
          <tr>
            <th><?=h(pretty_key($k))?></th>
            <td><?=nl2br(h((string)$v))?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </section>
  <?php endif; ?>

  <?php if ($records): ?>
  <section class="panel">
    <h2>Full Specifications (Table)</h2>
    <div class="scroll">
      <table class="rec">
        <thead>
          <tr><?php foreach ($cols as $c): ?><th><?=h(pretty_key((string)$c))?></th><?php endforeach; ?></tr>
        </thead>
        <tbody>
          <?php foreach ($records as $rec): ?>
            <tr>
              <?php foreach ($cols as $c): ?>
                <td><?=nl2br(h((string)($rec[$c] ?? '')))?></td>
              <?php endforeach; ?>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </section>
  <?php endif; ?>

  <?php if (!empty($row['specs_table_md'])): ?>
  <section class="panel">
    <h2>Markdown Specs (as captured)</h2>
    <pre class="code"><?=h($row['specs_table_md'])?></pre>
  </section>
  <?php endif; ?>
</main>

<footer class="wrap">
  <?php if (!empty($row['source_url'])): ?>
    Source: <a href="<?=h($row['source_url'])?>" target="_blank" rel="noopener"><?=h($row['source_url'])?></a>
  <?php endif; ?>
</footer>
</body>
</html>
