<?php
require __DIR__.'/config.php';

$pdo = db();

// ---- Inputs ----
$q      = trim($_GET['q']   ?? '');
$minp   = trim($_GET['min'] ?? '');
$maxp   = trim($_GET['max'] ?? '');
$sort   = trim($_GET['sort']?? 'newest');   // newest | price_asc | price_desc
$brand  = trim($_GET['brand']?? '');        // brand filter (from mapping_json)
$per    = min(48, max(12, (int)($_GET['per'] ?? 12)));
$page   = max(1, (int)($_GET['page'] ?? 1));

// ---- WHERE / Params ----
$where = [];
$params = [];

if ($q !== '') {
  $where[] = "(intro LIKE :q OR source_url LIKE :q OR mapping_json LIKE :q)";
  $params[':q'] = '%'.$q.'%';
}
if ($minp !== '' && ctype_digit($minp)) {
  $where[] = "price_rs IS NOT NULL AND price_rs >= :minp";
  $params[':minp'] = (int)$minp;
}
if ($maxp !== '' && ctype_digit($maxp)) {
  $where[] = "price_rs IS NOT NULL AND price_rs <= :maxp";
  $params[':maxp'] = (int)$maxp;
}
if ($brand !== '') {
  // case-insensitive text match inside mapping_json (brand or make fields)
  $where[] = "(LOWER(mapping_json) LIKE :brand_like)";
  $params[':brand_like'] = '%'.strtolower($brand).'%';
}

// ---- ORDER ----
$order = "ORDER BY created_at DESC, id DESC";
if ($sort === 'price_asc') {
  $order = "ORDER BY (price_rs IS NULL), price_rs ASC, id DESC";
} elseif ($sort === 'price_desc') {
  $order = "ORDER BY (price_rs IS NULL), price_rs DESC, id DESC";
}

$whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

// ---- Count ----
$stmt = $pdo->prepare("SELECT COUNT(*) FROM phones $whereSql");
$stmt->execute($params);
$total = (int)$stmt->fetchColumn();

$pages = max(1, (int)ceil($total / $per));
$page  = min($page, $pages);
$offset = ($page - 1) * $per;

// ---- Fetch rows ----
$sql = "SELECT id, source_url, remote_image_url, local_image_path, price_text, price_rs, intro, mapping_json, created_at
        FROM phones $whereSql $order LIMIT :lim OFFSET :off";
$stmt = $pdo->prepare($sql);
foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
$stmt->bindValue(':lim', $per, PDO::PARAM_INT);
$stmt->bindValue(':off', $offset, PDO::PARAM_INT);
$stmt->execute();
$rows = $stmt->fetchAll();

// ---- Build brand list (top 18 from recent 1000 rows) ----
$brandCounts = [];
try {
  $rs = $pdo->query("SELECT mapping_json FROM phones WHERE mapping_json IS NOT NULL AND mapping_json <> '' ORDER BY id DESC LIMIT 1000");
  $all = $rs->fetchAll(PDO::FETCH_COLUMN);
  foreach ($all as $mj) {
    $kv = json_decode($mj, true);
    if (!is_array($kv)) continue;
    $b = $kv['brand'] ?? ($kv['make'] ?? '');
    $b = trim((string)$b);
    if ($b === '') continue;
    // Normalize: First letter caps
    $b = mb_convert_case($b, MB_CASE_TITLE, 'UTF-8');
    $brandCounts[$b] = ($brandCounts[$b] ?? 0) + 1;
  }
} catch (Throwable $e) {
  // ignore brand panel on failure
}
arsort($brandCounts);
$brandsTop = array_slice(array_keys($brandCounts), 0, 18);

// ---- Helpers ----
function short_intro(?string $text): string {
  $text = (string)($text ?? '');
  if ($text === '') return '';
  return function_exists('mb_strimwidth')
    ? mb_strimwidth($text, 0, 160, '…', 'UTF-8')
    : (strlen($text) > 160 ? substr($text,0,157).'…' : $text);
}
function placeholder_svg(string $title): string {
  $t = htmlspecialchars($title ?: 'Phone', ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8');
  return 'data:image/svg+xml;utf8,'.
    rawurlencode('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 457" preserveAspectRatio="xMidYMid slice">
      <rect width="640" height="457" fill="#111111"/><text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle"
      font-family="system-ui,Segoe UI,Roboto,Arial" font-size="22" fill="#ffffff">'.$t.'</text></svg>');
}
function paginate_bootstrap(int $page, int $pages, array $qs = []): string {
  if ($pages <= 1) return '';
  $html = '<nav aria-label="Phones pagination"><ul class="pagination justify-content-center mt-4">';
  $mk = function($p, $label, $disabled=false, $active=false) use($qs){
    $qs['page'] = $p; $url = '?'.http_build_query($qs);
    $cls = 'page-item'; if ($disabled) $cls.=' disabled'; if ($active) $cls.=' active';
    return '<li class="'.$cls.'"><a class="page-link" href="'.h($url).'">'.h($label).'</a></li>';
  };
  $html .= $mk(max(1,$page-1), '« Prev', $page<=1, false);
  $window=2; $start=max(1,$page-$window); $end=min($pages,$page+$window);
  if ($start>1){ $html.=$mk(1,'1',false,$page===1); if($start>2)$html.='<li class="page-item disabled"><span class="page-link">…</span></li>'; }
  for($i=$start;$i<=$end;$i++){ $html.=$mk($i,(string)$i,false,$i===$page); }
  if ($end<$pages){ if($end<$pages-1)$html.='<li class="page-item disabled"><span class="page-link">…</span></li>'; $html.=$mk($pages,(string)$pages,false,$page===$pages); }
  $html .= $mk(min($pages,$page+1), 'Next »', $page>=$pages, false);
  return $html.'</ul></nav>';
}

// ---- SEO meta ----
$from = $total ? ($offset+1) : 0; $to = $total ? min($offset+$per,$total) : 0;
$siteName = 'MobileMart — Phones';
$metaDesc = $q ? "Search results for \"$q\" ($from–$to of $total)." : "Browse ".number_format($total)." phones. Filter by price & brand, sort by newest or price, and preview specs.";
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS']!=='off') ? 'https' : 'http';
$canonical = $scheme.'://'.$_SERVER['HTTP_HOST'].strtok($_SERVER['REQUEST_URI'],'?');
?>
<!doctype html>
<html lang="en" itemscope itemtype="https://schema.org/CollectionPage">
<head>
  <meta charset="utf-8">
  <title><?=h($siteName)?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="description" content="<?=h($metaDesc)?>">
  <link rel="canonical" href="<?=h($canonical)?>">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    :root{
      --ink:#0b0b0b; --ink-2:#111; --paper:#ffffff; --soft:#f6f7f9; --line:#e5e7eb; --muted:#9ca3af;
    }
    body{ color:var(--ink-2); background:linear-gradient(180deg,#fafafa, #fff 180px); }

    /* NAVBAR */
    .mm-navbar{ background:var(--ink); }
    .mm-navbar .navbar-brand{ font-weight:800; letter-spacing:.4px; color:#fff; }
    .mm-navbar .nav-link{ color:#fff; opacity:.88; }
    .mm-navbar .nav-link:hover{ opacity:1; text-decoration:underline; }
    .mm-search .form-control{
      background:#111; border:1px solid #fff; color:#fff; border-radius:999px; padding:.55rem .9rem;
    }
    .mm-search .form-control::placeholder{ color:#9aa0a6; }
    .mm-search .btn{ background:#fff; color:#000; border-color:#fff; border-radius:999px; padding:.55rem .9rem; }

    /* HERO */
    .hero{ background:linear-gradient(180deg,#000,#111); color:#fff; padding:56px 0 44px; position:relative; }
    .hero .title{ font-weight:800; letter-spacing:.2px; }
    .hero .subtitle{ color:#d1d5db; max-width:720px; }

    /* BRANDS */
    .brand-pills{ gap:.5rem; overflow:auto; white-space:nowrap; padding-bottom:.25rem; scrollbar-width:none; }
    .brand-pills::-webkit-scrollbar{ display:none; }
    .brand-pill{
      border:1px solid #e5e7eb; background:#fff; color:#111;
      padding:.4rem .75rem; border-radius:999px; text-decoration:none; white-space:nowrap; transition:.15s ease;
    }
    .brand-pill.active, .brand-pill:hover{ background:#000; color:#fff; border-color:#000; transform:translateY(-1px); }

    /* FILTERS */
    .filters-card{ border-radius:16px; background:#fff; }
    .mm-search-wide .mm-search-input{ background:#fff; color:#111; border-color:#ddd; }
    .mm-search-wide .mm-search-input:focus{ border-color:#bbb; box-shadow:none; }
    .mm-chip{
      display:inline-flex; align-items:center; gap:.45rem;
      background:#111; color:#fff; border-radius:999px; padding:.35rem .7rem; text-decoration:none; font-size:.9rem;
    }
    .mm-chip span{ opacity:.7; }
    .mm-chip:hover{ color:#fff; opacity:.95; }
    .mm-chip-clear{ background:#fff; color:#111; border:1px solid #111; }
    .mm-quick{
      border:1px solid #e5e7eb; background:#fff; color:#111;
      padding:.3rem .6rem; border-radius:999px; text-decoration:none; font-size:.9rem; transition:.15s;
    }
    .mm-quick.active, .mm-quick:hover{ background:#000; color:#fff; border-color:#000; }
    .mm-seg{ display:inline-flex; border:1px solid #e5e7eb; border-radius:999px; overflow:hidden; }
    .mm-seg a{ text-decoration:none; color:#111; padding:.45rem .8rem; background:#fff; border-right:1px solid #e5e7eb; }
    .mm-seg a:last-child{ border-right:none; }
    .mm-seg a.active{ background:#000; color:#fff; }
    .mm-per{ min-width:92px; }

    /* PROMO */
    .promo{ background:#000; color:#fff; border-radius:16px; padding:26px; }
    .promo .lead{ color:#e5e7eb; margin:0; }

    /* PRODUCT CARD */
    .card-product{ border-radius:16px; overflow:hidden; border:1px solid #eef0f2; transition:transform .12s ease, box-shadow .12s ease; background:#fff; }
    .card-product:hover{ transform:translateY(-2px); box-shadow:0 12px 24px rgba(0,0,0,.07); }
    .product-thumb{ background:#fff; display:flex; align-items:center; justify-content:center; border-bottom:1px solid #f0f0f0; }
    .product-thumb .ratio{ width:100%; }
    .product-thumb img{ object-fit:contain; }
    .price{ font-weight:800; }
    .badge-spec{ background:#f2f4f7; border:1px solid #e5e7eb; color:#111; }

    /* FOOTER */
    .mm-footer{ background:#000; color:#fff; }
    .mm-footer a{ color:#fff; text-decoration:none; opacity:.95; transition:.2s; }
    .mm-footer a:hover{ opacity:1; text-decoration:underline; }
    .mm-footer-hr{ border-top:1px solid rgba(255,255,255,.12); }
    .mm-footer-brand{ font-weight:800; letter-spacing:.3px; font-size:1.15rem; color:#fff; }
    .mm-footer-tagline{ color:#d1d5db; }
    .mm-footer-links li{ margin:.4rem 0; }
    .payment-badge{ display:inline-flex; align-items:center; gap:.5rem; padding:.45rem .7rem; border-radius:.6rem; background:#0f0f10; border:1px solid rgba(255,255,255,.18); color:#fff; font-size:.92rem; }
    .mm-play-badge{ border-color:#fff; color:#fff; border-radius:999px; padding:.45rem .8rem; transition:.25s; }
    .mm-play-badge:hover{ background:#fff; color:#000; }
    .social-icon{ width:40px; height:40px; border-radius:999px; display:inline-flex; align-items:center; justify-content:center; border:1px solid rgba(255,255,255,.35); color:#fff; transition:.25s; }
    .social-icon:hover{ background:#fff; color:#000; }
    .mm-footer ul{ margin:0; padding:0; }

    /* Pagination */
    .page-link{ color:#111; }
    .page-item.active .page-link{ background:#000; border-color:#000; }
  </style>
</head>
<body>

<!-- NAVBAR -->
<nav class="navbar navbar-expand-lg mm-navbar sticky-top">
  <div class="container">
    <a class="navbar-brand d-flex align-items-center gap-2" href="index.php"><i class="bi bi-phone"></i><span>MobileMart</span></a>

    <button class="navbar-toggler bg-light" type="button" data-bs-toggle="collapse" data-bs-target="#nav" aria-controls="nav" aria-expanded="false" aria-label="Toggle navigation">
      <span class="navbar-toggler-icon"></span>
    </button>

    <div id="nav" class="collapse navbar-collapse align-items-center">
      <ul class="navbar-nav me-lg-3 mb-2 mb-lg-0">
        <li class="nav-item"><a class="nav-link" href="?">Home</a></li>
        <li class="nav-item"><a class="nav-link" href="#brands">Brands</a></li>
        <li class="nav-item"><a class="nav-link" href="#filters">Filters</a></li>
      </ul>

      <div class="mm-search ms-auto me-auto w-100 my-2 my-lg-0" style="max-width:720px;">
        <form class="d-flex w-100" method="get" action="index.php" role="search" aria-label="Navbar search">
          <div class="input-group">
            <input class="form-control" type="search" name="q" value="<?=h($q)?>" placeholder="Search phones, specs, or URL">
            <button class="btn" type="submit"><i class="bi bi-search"></i></button>
          </div>
          <?php if ($brand!==''): ?><input type="hidden" name="brand" value="<?=h($brand)?>"><?php endif; ?>
          <?php if ($minp!==''):  ?><input type="hidden" name="min"   value="<?=h($minp)?>"><?php endif; ?>
          <?php if ($maxp!==''):  ?><input type="hidden" name="max"   value="<?=h($maxp)?>"><?php endif; ?>
          <?php if ($sort!=='newest'): ?><input type="hidden" name="sort"  value="<?=h($sort)?>"><?php endif; ?>
          <?php if ($per!==12):   ?><input type="hidden" name="per"   value="<?=h((string)$per)?>"><?php endif; ?>
        </form>
      </div>
    </div>
  </div>
</nav>

<!-- HERO -->
<section class="hero">
  <div class="container">
    <div class="row align-items-center g-4">
      <div class="col-lg-8">
        <h1 class="title display-5 mb-2">Find Your Next Phone — Fast.</h1>
        <p class="subtitle mb-4">Minimal, focused catalog. Filter by brand and price, sort by newest or price, and open a clean preview with the most relevant specs.</p>
        <div class="d-flex gap-2">
          <a href="#filters" class="btn btn-outline-light">Filter Phones</a>
          <a href="#grid" class="btn btn-light text-black">Browse All</a>
        </div>
      </div>
      <div class="col-lg-4 text-lg-end">
        <div class="small text-white-50">Showing <strong><?=number_format($total)?></strong> products</div>
      </div>
    </div>
  </div>
</section>

<!-- BRANDS -->
<section id="brands" class="py-4">
  <div class="container">
    <div class="d-flex align-items-center justify-content-between mb-2">
      <h2 class="h5 m-0">Popular Brands</h2>
      <a class="text-decoration-none" href="?">Clear</a>
    </div>
    <div class="d-flex brand-pills">
      <?php if (!empty($brandsTop)): ?>
        <?php foreach ($brandsTop as $b):
          $isActive = (strcasecmp($brand,$b)===0);
          $qs = $_GET; $qs['brand'] = $b; unset($qs['page']); $url = '?'.http_build_query($qs);
        ?>
          <a class="brand-pill <?=$isActive?'active':''?>" href="<?=h($url)?>"><?=h($b)?></a>
        <?php endforeach; ?>
      <?php else: ?>
        <span class="text-muted">Brands will appear when data is available.</span>
      <?php endif; ?>
    </div>
  </div>
</section>

<?php
// ---- Helpers for URLs (keep page=1 on any change) ----
$mkUrl = function(array $overrides = [], array $unset = []) {
  $qs = $_GET;
  foreach ($unset as $u) { unset($qs[$u]); }
  foreach ($overrides as $k=>$v) {
    if ($v === null || $v === '') unset($qs[$k]); else $qs[$k] = $v;
  }
  unset($qs['page']); // reset pagination when filters change
  return '?'.http_build_query($qs);
};

// ---- Global price stats (for quick ranges) ----
$stats = $pdo->query("SELECT MIN(price_rs) AS mn, MAX(price_rs) AS mx FROM phones WHERE price_rs IS NOT NULL")->fetch();
$gMin = (int)($stats['mn'] ?? 0);
$gMax = (int)($stats['mx'] ?? 0);

// Suggested quick ranges
$quickRanges = [
  ['label'=>'Under 30k', 'min'=>'',      'max'=>'30000'],
  ['label'=>'30k–60k',   'min'=>'30000', 'max'=>'60000'],
  ['label'=>'60k–100k',  'min'=>'60000', 'max'=>'100000'],
  ['label'=>'100k+',     'min'=>'100000','max'=>''],
];

// Active filter chips
$activeChips = [];
if ($q !== '')         $activeChips[] = ['label'=>"Search: $q",                    'url'=>$mkUrl([], ['q'])];
if ($brand !== '')     $activeChips[] = ['label'=>"Brand: $brand",                  'url'=>$mkUrl([], ['brand'])];
if ($minp !== '')      $activeChips[] = ['label'=>"Min: Rs ".number_format((int)$minp),'url'=>$mkUrl([], ['min'])];
if ($maxp !== '')      $activeChips[] = ['label'=>"Max: Rs ".number_format((int)$maxp),'url'=>$mkUrl([], ['max'])];
if ($sort!=='newest')  $activeChips[] = ['label'=>($sort==='price_asc'?'Sort: Price ↑':'Sort: Price ↓'),'url'=>$mkUrl(['sort'=>'newest'])];
if ($per!==12)         $activeChips[] = ['label'=>"Per page: $per",                 'url'=>$mkUrl(['per'=>12])];
?>

<!-- FILTERS -->
<section id="filters" class="py-3">
  <div class="container">
    <div class="card filters-card shadow-sm border-0">
      <div class="card-body p-3 p-md-4">

        <!-- Top row: centered search + result summary -->
        <div class="d-flex flex-column flex-md-row align-items-md-center justify-content-between gap-3 mb-3">
          <form class="mm-search-wide d-flex w-100" method="get" action="index.php" role="search" aria-label="Filters search">
            <div class="input-group" style="max-width:840px; margin:auto;">
              <input class="form-control mm-search-input" type="search" name="q" value="<?=h($q)?>" placeholder="Search phones, specs, or URL">
              <button class="btn btn-dark" type="submit"><i class="bi bi-search"></i></button>
            </div>
            <?php if ($brand!==''): ?><input type="hidden" name="brand" value="<?=h($brand)?>"><?php endif; ?>
            <?php if ($minp!==''):  ?><input type="hidden" name="min"   value="<?=h($minp)?>"><?php endif; ?>
            <?php if ($maxp!==''):  ?><input type="hidden" name="max"   value="<?=h($maxp)?>"><?php endif; ?>
            <?php if ($sort!=='newest'): ?><input type="hidden" name="sort" value="<?=h($sort)?>"><?php endif; ?>
            <?php if ($per!==12):   ?><input type="hidden" name="per"   value="<?=h((string)$per)?>"><?php endif; ?>
          </form>

          <div class="text-muted small text-md-end">
            <?php if ($total): ?>
              Showing <strong><?=$from?>–<?=$to?></strong> of <strong><?=number_format($total)?></strong>
              <?php if ($q !== ''): ?> • for “<strong><?=h($q)?></strong>”<?php endif; ?>
              <?php if ($brand !== ''): ?> • brand “<strong><?=h($brand)?></strong>”<?php endif; ?>
            <?php else: ?>
              No results yet — try adjusting filters
            <?php endif; ?>
          </div>
        </div>

        <!-- Active chips -->
        <?php if ($activeChips): ?>
          <div class="d-flex flex-wrap gap-2 mb-3">
            <?php foreach ($activeChips as $chip): ?>
              <a class="mm-chip" href="<?=h($chip['url'])?>"><?=h($chip['label'])?> <span aria-hidden="true">×</span></a>
            <?php endforeach; ?>
            <a class="mm-chip mm-chip-clear" href="?"><i class="bi bi-x-circle"></i> Clear all</a>
          </div>
        <?php endif; ?>

        <!-- Controls row -->
        <form class="row g-3 align-items-end" method="get" action="index.php">
          <!-- Brand scroller -->
          <div class="col-12">
            <label class="form-label text-muted">Brand</label>
            <div class="d-flex brand-pills overflow-auto gap-2 pb-1">
              <?php
                $urlAll = $mkUrl([], ['brand']);
                $isAll  = ($brand==='');
              ?>
              <a href="<?=h($urlAll)?>" class="brand-pill <?=$isAll?'active':''?>">All</a>
              <?php foreach ($brandsTop as $b):
                $urlB = $mkUrl(['brand'=>$b]);
                $isActive = (strcasecmp($brand,$b)===0);
              ?>
                <a href="<?=h($urlB)?>" class="brand-pill <?=$isActive?'active':''?>"><?=h($b)?></a>
              <?php endforeach; ?>
            </div>
          </div>

          <!-- Price + quick ranges -->
          <div class="col-12 col-md-6">
            <label class="form-label text-muted">Price (Rs)</label>
            <div class="row g-2">
              <div class="col-6">
                <input type="text" inputmode="numeric" pattern="[0-9]*" class="form-control" name="min" value="<?=h($minp)?>" placeholder="<?= $gMin ? number_format($gMin) : 'Min' ?>">
              </div>
              <div class="col-6">
                <input type="text" inputmode="numeric" pattern="[0-9]*" class="form-control" name="max" value="<?=h($maxp)?>" placeholder="<?= $gMax ? number_format($gMax) : 'Max' ?>">
              </div>
            </div>
            <div class="d-flex flex-wrap gap-2 mt-2">
              <?php foreach ($quickRanges as $r):
                $urlR = $mkUrl(['min'=>$r['min'],'max'=>$r['max']]);
                $on = ($minp===$r['min'] && $maxp===$r['max']);
              ?>
                <a class="mm-quick <?=$on?'active':''?>" href="<?=h($urlR)?>"><?=h($r['label'])?></a>
              <?php endforeach; ?>
            </div>
          </div>

          <!-- Sort segmented + Per page -->
          <div class="col-12 col-md-6">
            <label class="form-label text-muted">Sort & Per Page</label>
            <div class="d-flex flex-wrap gap-2">
              <div class="mm-seg">
                <?php
                  $uNewest = $mkUrl(['sort'=>'newest']);
                  $uAsc    = $mkUrl(['sort'=>'price_asc']);
                  $uDesc   = $mkUrl(['sort'=>'price_desc']);
                ?>
                <a href="<?=h($uNewest)?>" class="<?= $sort==='newest'?'active':'' ?>">Newest</a>
                <a href="<?=h($uAsc)?>"    class="<?= $sort==='price_asc'?'active':'' ?>">Price ↑</a>
                <a href="<?=h($uDesc)?>"   class="<?= $sort==='price_desc'?'active':'' ?>">Price ↓</a>
              </div>
              <div>
                <select class="form-select mm-per" name="per" onchange="this.form.submit()">
                  <option value="12" <?=$per===12?'selected':''?>>12</option>
                  <option value="24" <?=$per===24?'selected':''?>>24</option>
                  <option value="48" <?=$per===48?'selected':''?>>48</option>
                </select>
              </div>
              <div class="ms-auto">
                <button class="btn btn-dark" type="submit"><i class="bi bi-funnel"></i> Apply</button>
                <a class="btn btn-outline-secondary" href="?">Reset</a>
              </div>
            </div>
          </div>

          <!-- Preserve query on submit -->
          <?php if ($q!==''):    ?><input type="hidden" name="q" value="<?=h($q)?>"><?php endif; ?>
          <?php if ($brand!==''): ?><input type="hidden" name="brand" value="<?=h($brand)?>"><?php endif; ?>
        </form>

      </div>
    </div>
  </div>
</section>

<!-- PROMO BANNER -->
<section class="py-4">
  <div class="container">
    <div class="promo d-flex flex-column flex-md-row align-items-center justify-content-between">
      <div class="mb-3 mb-md-0">
        <h3 class="h4 mb-1">Weekly Spotlight</h3>
        <p class="lead">Handpicked phones with great value. Check specs and compare prices in seconds.</p>
      </div>
      <a href="#grid" class="btn btn-outline-light mt-2 mt-md-0">Browse Now</a>
    </div>
  </div>
</section>

<!-- PRODUCT GRID -->
<section id="grid" class="pb-4">
  <div class="container">
    <?php if (!$rows): ?>
      <div class="alert alert-light border text-center py-5">
        <div class="mb-2"><i class="bi bi-emoji-frown" style="font-size:2rem;"></i></div>
        Try broadening your search or clearing filters.
      </div>
    <?php else: ?>
      <div class="row g-3 g-md-4">
        <?php foreach ($rows as $r):
          $title  = derive_title($r);
          $imgUrl = product_image_url($r);
          $badges = top_specs($r);
          $detail = 'preview.php?id='.$r['id'];
          $priceText = money($r);
          $img = $imgUrl ?: placeholder_svg($title);
          $isDataUri = str_starts_with($img, 'data:image');
        ?>
        <div class="col-12 col-sm-6 col-md-4 col-lg-3 d-flex">
          <article class="card card-product flex-fill" itemscope itemtype="https://schema.org/Product">
            <a class="product-thumb text-center" href="<?=h($detail)?>" aria-label="<?=h($title)?>">
              <div class="ratio ratio-4x3">
                <img src="<?=h($img)?>" <?= $isDataUri ? '' : 'loading="lazy" decoding="async"' ?> alt="<?=h($title)?>" class="img-fluid" itemprop="image">
              </div>
            </a>
            <div class="card-body d-flex flex-column">
              <h3 class="h6 mb-1" itemprop="name"><?=h($title)?></h3>
              <div class="price mb-2" itemprop="offers" itemscope itemtype="https://schema.org/Offer">
                <meta itemprop="priceCurrency" content="PKR">
                <span itemprop="price"><?=h($priceText)?></span>
              </div>
              <?php if (!empty($r['intro'])): ?>
                <p class="text-muted small mb-2"><?=h(short_intro($r['intro']))?></p>
              <?php endif; ?>
              <?php if ($badges): ?>
                <div class="d-flex flex-wrap gap-1 mb-3" aria-label="Key specs">
                  <?php foreach ($badges as $b): ?>
                    <span class="badge rounded-pill badge-spec"><?=h($b)?></span>
                  <?php endforeach; ?>
                </div>
              <?php endif; ?>
              <div class="mt-auto d-flex gap-2">
                <a class="btn btn-outline-dark flex-fill" href="<?=h($r['source_url'])?>" target="_blank" rel="noopener">
                  <i class="bi bi-box-arrow-up-right"></i> Source
                </a>
                <a class="btn btn-dark flex-fill" href="<?=h($detail)?>">
                  <i class="bi bi-eye"></i> Preview
                </a>
              </div>
            </div>
          </article>
        </div>
        <?php endforeach; ?>
      </div>

      <?=paginate_bootstrap($page, $pages, ['q'=>$q,'min'=>$minp,'max'=>$maxp,'sort'=>$sort,'per'=>$per,'brand'=>$brand])?>
    <?php endif; ?>
  </div>
</section>

<!-- FOOTER -->
<footer class="mm-footer pt-5 pb-4 mt-4">
  <div class="container">
    <div class="row gy-4 align-items-start">
      <div class="col-md-4">
        <a class="mm-footer-brand d-inline-flex align-items-center mb-2" href="index.php">
          <i class="bi bi-phone me-2"></i> <span>MobileMart</span>
        </a>
        <p class="mm-footer-tagline mb-3">Minimal, fast and focused smartphone catalog.</p>
        <a href="#" class="btn btn-outline-light btn-sm mm-play-badge">
          <i class="bi bi-google-play me-1"></i> Get it on Google Play
        </a>
      </div>

      <div class="col-6 col-md-4 col-lg-3">
        <h6 class="text-uppercase fw-bold mb-2">Customer Service</h6>
        <ul class="list-unstyled mm-footer-links">
          <li><a href="#">Help Center</a></li>
          <li><a href="#">Privacy Policy</a></li>
          <li><a href="#">Installments Plan</a></li>
          <li><a href="#">E-Warranty Activation</a></li>
          <li><a href="#">Sell on MobileMart</a></li>
        </ul>
      </div>

      <div class="col-6 col-md-4 col-lg-5">
        <h6 class="text-uppercase fw-bold mb-2">Secure Payment Methods</h6>
        <div class="d-flex flex-wrap gap-2 mb-3">
          <span class="payment-badge"><i class="bi bi-credit-card-2-front"></i> VISA</span>
          <span class="payment-badge"><i class="bi bi-credit-card"></i> MasterCard</span>
          <span class="payment-badge"><i class="bi bi-bank"></i> HBL</span>
          <span class="payment-badge"><i class="bi bi-cash-coin"></i> Cash on Delivery</span>
          <span class="payment-badge"><i class="bi bi-link-45deg"></i> 1LINK</span>
        </div>
      </div>
    </div>

    <hr class="mm-footer-hr my-4">

    <div class="row gy-3 align-items-center">
      <div class="col-md-6">
        <div class="d-flex gap-2">
          <a class="social-icon" href="#" aria-label="YouTube"><i class="bi bi-youtube"></i></a>
          <a class="social-icon" href="#" aria-label="Facebook"><i class="bi bi-facebook"></i></a>
          <a class="social-icon" href="#" aria-label="Instagram"><i class="bi bi-instagram"></i></a>
          <a class="social-icon" href="#" aria-label="TikTok"><i class="bi bi-tiktok"></i></a>
          <a class="social-icon" href="#" aria-label="LinkedIn"><i class="bi bi-linkedin"></i></a>
        </div>
      </div>
      <div class="col-md-6 text-md-end small text-white-50">
        Copyright © <?=date('Y')?> MobileMart. All rights reserved.
      </div>
    </div>

    <div class="mm-footer-copy mt-3 small text-white-50">
      <p>Prices are in PKR. Specs and availability are sourced from partner pages and may change without notice. Please verify details before purchase.</p>
    </div>
  </div>
</footer>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
// numeric-only soft guard for min/max
document.querySelectorAll('input[name="min"],input[name="max"]').forEach(function(el){
  el.addEventListener('input', function(){ this.value = this.value.replace(/[^0-9]/g,''); });
});
</script>
</body>
</html>
